/***

Copyright (c) 2008, Agilent Technologies

Module Name:

    b_lxpci.c	(Best-Analyser Linux PCI Device Driver)

Abstract:

    A driver which creates (and adds to the registry) one device for each PCI
    function existing when loaded.

Notes:
  Additional functionality of each device includes Read/Write
  of I/O and Config. space.

***/

#include <linux/autoconf.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/stat.h>
#include <b_pci.h>
#include <b_ioctl.h>
#include <asm/uaccess.h>
#include <linux/version.h>

MODULE_LICENSE("GPL");

// The majority of the code in here is only used during DriverEntry() so
// we can discard some functions after use.  We MUST declare these
// here and define them AFTER the ALLOC_PRAGMA.  (!!!!)

#if DEBUG
#define DBG_PRINT(arg...) printk(KERN_INFO arg)
#else
#define DBG_PRINT(arg...)
#endif

static struct file_operations bestpci_fops = {
    owner:      THIS_MODULE,
    read:       bestpci_read,
    write:      bestpci_write,
    ioctl:      bestpci_ioctl,
    open:       bestpci_open,
    release:    bestpci_close
};

static struct pci_device_id agt_pci_ids[] = {
	{ PCI_DEVICE(AGILENT_VENDOR_ID, AGILENT_DEVICE_2960) },
	{ 0, },
};
MODULE_DEVICE_TABLE(pci, agt_pci_ids);

static int numdevices=1;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,21) // kernel 2.6.21 uses this
static struct class *bestpci_device_class;
#else
static struct class_simple *bestpci_device_class; // 2.6.9 use this
#endif

static struct pci_dev *bestpci_pci_devices[256];

static struct pci_driver bestpci_driver = {
    name:       "BestPCI",
    id_table:   agt_pci_ids,
    probe:      bestpci_probe,
    remove:     bestpci_remove,
};

#if CUSTOM_OEM1
/* Why is this global/global ? */
spinlock_t CAPI_Lock;
#endif

#if CUSTOM_OEM1
MODULE_DESCRIPTION("Agilent Linux support driver\n  made on "
				   __DATE__ " at " __TIME__);
#endif

// protoypes for linux version dependent calls
static void bestpci_create_class(void);
static void bestpci_destroy_class(void);
static struct class_device* bestpci_create_class_device(int minor);
static void bestpci_destroy_class_device(int minor);

/////////////////////////////////////////////////////////////////////////
// kernel module calls (init, exit)
/////////////////////////////////////////////////////////////////////////
static int __init bestpci_init_module(void) {
  // First register major-device
  int status=0;
  struct class_device *theDevice = NULL;
  DBG_PRINT("b_lxpci: bestpci_init_module called\n");

#if CUSTOM_OEM1
  spin_lock_init(&CAPI_Lock);
#endif

  // Create a class bestpci. This will add an entry in the sysfs
  // named /sys/class/bestpci. 
  DBG_PRINT("b_lxpci: class_create\n");

  // this is linux kernel version dependent
  bestpci_create_class();

  if (bestpci_device_class == NULL || IS_ERR(bestpci_device_class)) {
    printk(KERN_ERR "b_lxpci: unable to create class bestpci_device_class\n");
    return PTR_ERR(bestpci_device_class);
  }

  // Register the char device driver
  DBG_PRINT("b_lxpci: calling register_chrdev\n");
  status=register_chrdev(BESTPCI_MAJOR, "bestpci", &bestpci_fops);
  if (status != 0) {
    printk(KERN_ERR "b_lxpci: unable to register char-device %d - %d\n",BESTPCI_MAJOR,status);
    bestpci_destroy_class();
    return status;
  }

  // Register the pci driver
  DBG_PRINT("b_lxpci: pci_register_driver\n");
  status=pci_register_driver(&bestpci_driver);
  if (status < 0) {
    printk(KERN_ERR "b_lxpci: unable to register pci-driver, %d\n",status);
    unregister_chrdev(BESTPCI_MAJOR,"bestpci");
    bestpci_destroy_class();
    return status;
  } 

  // Create a device for the class bestpci_device_class. This will add a 'dev' entry
  // within /sys/class/bestpci containing major:minor number. This entry is needed 
  // by udev to automatically create the \dev\bestpci0 entry. The '0' entry is the 
  // generic device used by the PortAPI to figure out the version number and the 
  // number of devices connected to the system.
  DBG_PRINT("b_lxpci: class_device_create\n"); 

  theDevice = bestpci_create_class_device(0);
  if(theDevice == NULL || IS_ERR(theDevice))
  { 
    printk(KERN_ERR "b_lxpci: error calling class_device_create: %ld.\n",PTR_ERR(theDevice));
    pci_unregister_driver(&bestpci_driver);
    unregister_chrdev(BESTPCI_MAJOR,"bestpci");
    bestpci_destroy_class();
    return PTR_ERR(theDevice);
  }

  return status; 
}

//*************************************************************************
static __exit void bestpci_exit_module(void) {
  int i = 0;
  pci_unregister_driver(&bestpci_driver);
  unregister_chrdev(BESTPCI_MAJOR,"bestpci"); 
  for(i = 0; i < numdevices ; i++)
  {
    bestpci_destroy_class_device(i);
    DBG_PRINT("b_lxpci: destroying class device %i\n",i);
  }
  bestpci_destroy_class();
  DBG_PRINT("b_lxpci: successfully removed driver\n");
}


/////////////////////////////////////////////////////////////////////////
// driver callback calls (probe, remove)
/////////////////////////////////////////////////////////////////////////
static int __devinit bestpci_probe(struct pci_dev *dev, const struct pci_device_id *id) {
    char name[7];
    int i;
    struct class_device *theDevice = NULL;
    bestpci_device_info *devinfo = (bestpci_device_info *) kmalloc(sizeof(bestpci_device_info),GFP_KERNEL);

    printk(KERN_INFO "b_lxpci: bestpci_probe called\n");
    sprintf(name,"pci%d",numdevices);

    bestpci_pci_devices[numdevices]=dev;

    devinfo->enumdevice = numdevices;

    devinfo->f_device_open = FALSE;
    devinfo->f_is_hp_best_device = FALSE;

#if CUSTOM_OEM1
    spin_lock_init(&(devinfo->lock));
#endif

    DBG_PRINT("b_lxpci: VendorID: %x - DeviceID: %x\n",dev->vendor,dev->device);

    if (dev->vendor == HP_VENDOR_ID) {
      switch (dev->device) {
        case HP_DEVICE_2925:
        case HP_DEVICE_2926:
        case HP_DEVICE_2927:
        case HP_DEVICE_2928:
        case HP_DEVICE_2940:
            devinfo->f_is_hp_best_device=TRUE;
            devinfo->f_is_2926=TRUE;
            DBG_PRINT("b_lxpci: is hp-best-device\n");
            break;
      }
    }
    if (dev->vendor == AGILENT_VENDOR_ID) {
      switch (dev->device) {
        case AGILENT_DEVICE_2922:
        case AGILENT_DEVICE_2929:
            devinfo->f_is_hp_best_device=TRUE;
            devinfo->f_is_2926=TRUE;
            DBG_PRINT("b_lxpci: is agilent-best-device\n");
            break;
      }
    }

    devinfo->device_id = dev->device;
    devinfo->vendor_id = dev->vendor;
    devinfo->sub_vendor_id = dev->subsystem_vendor;
    devinfo->sub_device_id = dev->subsystem_device;
    devinfo->class_ = dev->class;

    devinfo->baseaddress = (void *)INVALID_BASE_IOADDR;
    devinfo->decoderstatus = DECODER_DISABLED;
    devinfo->f_using_direct_io = FALSE;

    if (devinfo->f_is_hp_best_device) {
      i=0;
      while ( (!(pci_resource_flags(dev,i)&IORESOURCE_IO)) && (i<5)) {
        i++;
      }
      if (i<5) {
        devinfo->baseaddress = ioremap(pci_resource_start(dev,i),pci_resource_len(dev,i));
        devinfo->baselen = pci_resource_len(dev,i);
        request_region((unsigned long)devinfo->baseaddress,devinfo->baselen,name);
        devinfo->f_using_direct_io = TRUE;
        devinfo->decoderstatus = DECODER_ENABLED | DECODER_LOADED;
      }
      DBG_PRINT("b_lxpci: registered io-address 0x%p length %d\n",devinfo->baseaddress,devinfo->baselen);
    }

    devinfo->f_using_direct_io = FALSE;

    devinfo->devicetype = NTIO_TYPE_PCI;
    devinfo->regwidth = 1;
    devinfo->commtimeouts = defTimeouts;
    devinfo->slot_id.u.bits.FunctionNumber = PCI_FUNC(dev->devfn);
    devinfo->slot_id.u.bits.DeviceNumber = PCI_SLOT(dev->devfn);
    devinfo->slot_id.u.bits.BusNumber = dev->bus->number;
    devinfo->slot_id.u.bits.Reserved = 0;

    pci_set_drvdata(dev,devinfo);
   
    // Create a device for the class bestpci_device_class. This will add a 'dev' entry
    // within /sys/class/bestpci containing major:minor number. This entry is needed 
    // by udev to automatically create the \dev\bestpci%d

    theDevice = bestpci_create_class_device(numdevices);

    if(theDevice == NULL || IS_ERR(theDevice))
    { 
      printk(KERN_ERR "b_lxpci: error adding class device: %ld.\n",PTR_ERR(theDevice));
    }

    DBG_PRINT("b_lxpci: registered device %s\n",name);

    numdevices++;
    return 0;
}

//*************************************************************************
static void bestpci_remove(struct pci_dev *dev) {
  bestpci_device_info *devinfo = pci_get_drvdata(dev);
  printk(KERN_INFO "b_lxpci: bestpci_remove started\n");
  if (devinfo->baseaddress != (void *)INVALID_BASE_IOADDR) {
    DBG_PRINT("b_lxpci: Releasing %p, devid %x, venid %x\n",devinfo->baseaddress,dev->device,dev->vendor);
    release_region((unsigned long)devinfo->baseaddress,devinfo->baselen);
    iounmap(devinfo->baseaddress);
  }
}

/////////////////////////////////////////////////////////////////////////
// file operation calls (open, close, write, read, ioctl)
/////////////////////////////////////////////////////////////////////////
static int bestpci_open(struct inode *inode, struct file *file) {
  int status=0;
  unsigned int minor = MINOR(inode->i_rdev);
  bestpci_device_info *devinfo;
  
#if CUSTOM_OEM1
  unsigned long flags ;
#endif

  if (minor>numdevices) {
    DBG_PRINT("b_lxpci: Error: Tried to open unknown device %d\n",minor);
    return -ENODEV;
  }

  if (minor==0) {
    return 0;
  }

  // Need to ensure we don't have competition for opening the card
#if CUSTOM_OEM1
  spin_lock_irqsave( &CAPI_Lock, flags ) ;
#endif

  devinfo=(bestpci_device_info *) pci_get_drvdata(bestpci_pci_devices[minor]);
  if (devinfo->f_device_open) {
    DBG_PRINT("b_lxpci: Error: Device %d already open...\n",minor);
#if CUSTOM_OEM1
    spin_unlock_irqrestore( &CAPI_Lock, flags ) ;
#endif
    return -EBUSY;
  }
  
  devinfo->f_device_open = TRUE;

#if CUSTOM_OEM1
  // Critical section for claiming a card done.
  spin_unlock_irqrestore( &CAPI_Lock, flags ) ;
#endif

  if (devinfo->f_is_hp_best_device) {
    status = pci_enable_device(bestpci_pci_devices[minor]);
    if (status < 0)
    {
      return status;
    }
    pci_set_master(bestpci_pci_devices[minor]);
    bestpci_disconnect(bestpci_pci_devices[minor]);
  }

  return 0;
}

//*************************************************************************
static int bestpci_close(struct inode *inode, struct file *file) {
  unsigned int minor = MINOR(inode->i_rdev);
  bestpci_device_info *devinfo;

#if CUSTOM_OEM1
  unsigned long flags ;
#endif

  if (minor==0) {
    return 0;
  }

#if CUSTOM_OEM1
  // Need to ensure we don't have competition for open/closing the card
  spin_lock_irqsave( &CAPI_Lock, flags ) ;
#endif

  devinfo=(bestpci_device_info *) pci_get_drvdata(bestpci_pci_devices[minor]);

  devinfo->f_device_open = FALSE;
  devinfo->f_using_direct_io = FALSE;

  bestpci_disconnect(bestpci_pci_devices[minor]);

#if CUSTOM_OEM1
  spin_unlock_irqrestore( &CAPI_Lock, flags ) ;
#endif

  return 0;
}

//*************************************************************************
static int bestpci_write(struct file *file, const char *buf, size_t count, loff_t *ppos) {
  unsigned int minor = MINOR(file->f_dentry->d_inode->i_rdev);
  bestpci_device_info *devinfo;
  struct pci_dev *dev;
  u32 data;
  size_t bytes_left;

#if CUSTOM_OEM1
  unsigned long flags ;
#endif

  // Need to avoid spin loops here on status bits -- add sleep or suspend?
  if (minor == 0) {
    return 0;
  } else {
    dev=bestpci_pci_devices[minor];
    devinfo=pci_get_drvdata(dev);
    bytes_left=count;
    if (devinfo->f_using_direct_io) {
      return -EINVAL; // Not yet supported
    } else {

#if CUSTOM_OEM1
      spin_lock_irqsave( &(devinfo->lock), flags ) ;
#endif
      while (bytes_left) {
        pci_read_config_dword(dev,PCI_STATUS_E2925A,&data);
        if (! (data & PCI_STATUS_E2925A_SEND_FULL_BIT)) {
          if(copy_from_user(&data,buf,sizeof(u8)))
            return -EFAULT;
          pci_write_config_byte(dev,PCI_DATA,data);
          buf += sizeof(u8);
          bytes_left -= sizeof(u8);
        } else {
#if CUSTOM_OEM1
      spin_unlock_irqrestore( &(devinfo->lock), flags ) ;
#endif
	  return count-bytes_left;
        }
      }
#if CUSTOM_OEM1
      spin_unlock_irqrestore( &(devinfo->lock), flags ) ;
#endif
      return count-bytes_left;
    }
  }

  return 0;
}

//*************************************************************************
static int bestpci_read(struct file *file, char *buf, size_t count, loff_t *ppos) {
  unsigned int minor = MINOR(file->f_dentry->d_inode->i_rdev);
  bestpci_device_info *devinfo;
  struct pci_dev *dev;
  u32 data;
  size_t bytes_left;
#if CUSTOM_OEM1
  unsigned long flags ;
#endif

  // Need to avoid spin loops here on status bits -- add sleep or suspend?
  if (minor == 0) {
    return 0;
  } else {
    dev=bestpci_pci_devices[minor];
    devinfo=pci_get_drvdata(dev);
    bytes_left=count;
    if (devinfo->f_using_direct_io) {
      return -EINVAL; // Not yet supported
    } else {
#if CUSTOM_OEM1
      spin_lock_irqsave( &(devinfo->lock), flags ) ;
#endif
      while (bytes_left) {
        pci_read_config_dword(dev,PCI_STATUS_E2925A,&data);
        if (data & PCI_STATUS_E2925A_DATA_VALID_BIT) {
          pci_read_config_byte(dev,PCI_DATA,(u8 *)(&data));
          if (copy_to_user(buf,&data,sizeof(u8)))
            return -EFAULT;
          buf += sizeof(u8);
          bytes_left -= sizeof(u8);
          data = PCI_STATUS_E2925A_DATA_VALID_BIT;
          pci_write_config_dword(dev,PCI_STATUS_E2925A,data);
        } else {
#if CUSTOM_OEM1
      spin_unlock_irqrestore( &(devinfo->lock), flags ) ;
#endif
	  return count-bytes_left;
	}
      }
#if CUSTOM_OEM1
      spin_unlock_irqrestore( &(devinfo->lock), flags ) ;
#endif
      return count-bytes_left;
    }
  }
  return 0;
}

//*************************************************************************
static int bestpci_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg) {
  unsigned int minor = MINOR(inode->i_rdev);
  struct pci_dev *dev=bestpci_pci_devices[minor];
  bestpci_device_info *devinfo;
  bestpci_general_info geninfo;
  void *ptr;
  int numBytes;

  printk(KERN_INFO "b_lxpci: bestpci_ioctl 2\n");

  if (minor == 0) {
    if (cmd==0) {
      geninfo.version=DEV_VERSION;
      geninfo.dev_count=numdevices-1;
      if (copy_to_user((void *)arg,&geninfo,sizeof(geninfo)))
        return -EFAULT;
     printk(KERN_INFO "b_lxpci: bestpci_ioctl minor and cmd is 0\n"); 
     return 0;
    } else {
      printk(KERN_INFO "b_lxpci: bestpci_ioctl cmd is %d\n",cmd); 
      devinfo=pci_get_drvdata(bestpci_pci_devices[cmd]);
      printk(KERN_INFO "b_lxpci: bestpci_ioctl cmd is not 0 level 1\n"); 
      numBytes = sizeof(*devinfo);
#ifdef CUSTOM_OEM1
// If the kernel module is compiled using CUSTOM_OEM1 option,
// bestpci_device_info has a member spinlock_t. This mustn't 
// be copied into userspace, as the spinlock_t member is not 
// part of the bestpci_device_info here. This would result in 
// writing into uninitialized memory in user space!!
      numBytes = numBytes - sizeof(spinlock_t);
#endif
//      printk(KERN_INFO "b_lxpci: devinfo copy size 0x%x\n",numBytes); 
      if (copy_to_user((void *)arg,devinfo,numBytes))
        return -EFAULT;
      printk(KERN_INFO "b_lxpci: bestpci_ioctl cmd is not 0 level 2\n"); 
    }
    return 0;
  } else {
    devinfo=pci_get_drvdata(dev);
    switch (cmd) {
      case IOCTL_NTIO_GET_CONFIG_DW:
          printk(KERN_INFO "b_lxpci: bestpci_ioctl IOCTL_NTIO_GET_CONFIG_DW\n"); 
	  ptr=kmalloc(sizeof(u32),GFP_KERNEL);
          if (copy_from_user(ptr,(void *)arg,sizeof(u32)))
            return -EFAULT;
          *((u32 *)ptr)=bestpci_read_config_dw(dev,*((u32 *)ptr));
          if (copy_to_user((void *)arg,ptr,sizeof(u32)))
            return -EFAULT;
          kfree(ptr);
          break;
      case IOCTL_NTIO_SET_CONFIG_DW:
          printk(KERN_INFO "b_lxpci: bestpci_ioctl IOCTL_NTIO_SET_CONFIG_DW\n");
          ptr=kmalloc(sizeof(b_configdrivertype),GFP_KERNEL); 
	  if (copy_from_user(ptr,(void *)arg,sizeof(b_configdrivertype)))
            return -EFAULT;
          bestpci_write_config_dw(dev,(b_configdrivertype *)ptr);
          kfree(ptr);
          break;
      case IOCTL_NTIO_READ_REG_DW:
          printk(KERN_INFO "b_lxpci: bestpci_ioctl IOCTL_NTIO_READ_REG_DW\n");
  	  ptr=malloc(sizeof(b_accessporttype));
          if (copy_from_user(ptr,(void *)arg,sizeof(b_accessporttype)))
            return -EFAULT;
          *((u32 *)ptr)=bestpci_direct_reg_read(dev,(b_accessporttype *)ptr);
          if (copy_to_user((void *)arg,ptr,sizeof(u32)))
            return -EFAULT;
          kfree(ptr);
          break;
      case IOCTL_NTIO_WRITE_REG_DW:
          printk(KERN_INFO "b_lxpci: bestpci_ioctl IOCTL_NTIO_WRITE_REG_DW\n");
	  ptr=malloc(sizeof(b_accessporttype));
          if (copy_from_user(ptr,(void *)arg,sizeof(b_accessporttype)))
            return -EFAULT;
          bestpci_direct_reg_write(dev,(b_accessportdatatype *)arg);
          kfree(ptr);
          break;
      case IOCTL_NTIO_SET_DIRECT_IO:
          printk(KERN_INFO "b_lxpci: bestpci_ioctl IOCTL_NTIO_SET_DIRECT_IO\n");
          if (DECODER_READY & devinfo->decoderstatus) {
            devinfo->f_using_direct_io = TRUE;
          } else {
            return -EINVAL;
          }
          break;
      case IOCTL_NTIO_SET_HAL_IO:
          printk(KERN_INFO "b_lxpci: bestpci_ioctl IOCTL_NTIO_SET_HAL_IO\n");
          devinfo->f_using_direct_io = FALSE;
          break;
      case IOCTL_NTIO_SET_TIMEOUT:
          printk(KERN_INFO "b_lxpci: bestpci_ioctl IOCTL_NTIO_SET_TIMEOUT\n");
  	  if (copy_from_user(&devinfo->commtimeouts, (BESTTIMEOUTS *)arg, sizeof(BESTTIMEOUTS)));
            return -EFAULT;
          break;
      case IOCTL_NTIO_SET_REGWIDTH_PCI:
          printk(KERN_INFO "b_lxpci: bestpci_ioctl IOCTL_NTIO_SET_REGWIDTH_PCI\n");
          devinfo->regwidth = arg;
          break;
      case IOCTL_NTIO_GET_BASE_ADDR:
          printk(KERN_INFO "b_lxpci: bestpci_ioctl IOCTL_NTIO_GET_BASE_ADDR\n");
          if (DECODER_LOADED & devinfo->decoderstatus) {
            if (copy_to_user((void *)arg,&devinfo->baseaddress,sizeof(u32)))
              return -EFAULT;
          } else {
            return -EINVAL;
          }
          break;
    }
    return 0;
  }
  return -1;
}

/////////////////////////////////////////////////////////////////////////
// Helper functions
/////////////////////////////////////////////////////////////////////////
void bestpci_disconnect(struct pci_dev *dev) {
  u32 data=0;
  bestpci_device_info *devinfo;

  devinfo=(bestpci_device_info *) pci_get_drvdata(dev);

  if (devinfo->f_is_hp_best_device) {
    pci_write_config_dword(dev,PCI_CONNECT_CMD,data);
  }
}

//*************************************************************************
u32 bestpci_direct_reg_read(struct pci_dev *dev,b_accessporttype *pPortData) {
  u8 bStatus;
  bestpci_device_info *devinfo = pci_get_drvdata(dev);
  u32 data;
#if CUSTOM_OEM1
  unsigned long flags ;
#endif

  // Need to avoid spin loops here on status bits -- add sleep or suspend?
  if (devinfo->f_using_direct_io) {
    return -1;
  } else {
#if CUSTOM_OEM1
    spin_lock_irqsave( &(devinfo->lock), flags ) ;
#endif
    do {
      pci_read_config_byte(dev,ACCESSPORT_STAT,&bStatus);
    } while (!(bStatus & ACCESSPORT_STAT_PORT_RDY));

    switch (pPortData->m_cRegWidth) {
        case 1UL: data=ACCESSPORT_CMD_SIZE_BYTE;
                  break;
        case 2UL: data=ACCESSPORT_CMD_SIZE_WORD;
                  break;
        case 4UL: data=ACCESSPORT_CMD_SIZE_DWORD;
                  break;
    }
    data|=ACCESSPORT_CMD_READ;
    data|=((pPortData->m_dwAddress << 8) & 0xFFFFFF00UL);
    pci_write_config_dword(dev,ACCESSPORT_CMD_ADDRESS,data);
    do {
      pci_read_config_byte(dev,ACCESSPORT_STAT,&bStatus);
    } while (!(bStatus & ACCESSPORT_STAT_DATA_RDY));
    pci_read_config_dword(dev,ACCESSPORT_DATA,&data);

#if CUSTOM_OEM1
    spin_unlock_irqrestore( &(devinfo->lock), flags ) ;
#endif
    return (data & (0xFFFFFFFF >> (8*(4-pPortData->m_cRegWidth))));/* fix for wrong shift value */ 
 
  }
}

//*************************************************************************
void bestpci_direct_reg_write(struct pci_dev *dev,b_accessportdatatype *pPortData) {
  u8 bStatus;
  bestpci_device_info *devinfo = pci_get_drvdata(dev);
  u32 data=0;
#if CUSTOM_OEM1
  unsigned long flags ;
#endif

  // Need to avoid spin loops here on status bits -- add sleep or suspend?
  if (devinfo->f_using_direct_io) {
    return;
  } else {
#if CUSTOM_OEM1
    spin_lock_irqsave( &(devinfo->lock), flags ) ;
#endif
    do {
      pci_read_config_byte(dev,ACCESSPORT_STAT,&bStatus);
    } while (!(bStatus & ACCESSPORT_STAT_PORT_RDY));

    switch (pPortData->m_port.m_cRegWidth) {
        case 1UL: data=ACCESSPORT_CMD_SIZE_BYTE;
                  break;
        case 2UL: data=ACCESSPORT_CMD_SIZE_WORD;
                  break;
        case 4UL: data=ACCESSPORT_CMD_SIZE_DWORD;
                  break;
    }
    data&=~ACCESSPORT_CMD_READ;
    data|=((pPortData->m_port.m_dwAddress << 8) & 0xFFFFFF00UL);
    pci_write_config_dword(dev,ACCESSPORT_CMD_ADDRESS,data);
    pci_write_config_dword(dev,ACCESSPORT_DATA,pPortData->m_data);

#if CUSTOM_OEM1
    spin_unlock_irqrestore( &(devinfo->lock), flags ) ;
#endif
  }
}

//*************************************************************************
u32 bestpci_read_config_dw(struct pci_dev *dev,u32 data) {
  bestpci_device_info *devinfo = pci_get_drvdata(dev);
  u32 data2;

  if (devinfo->f_using_direct_io) {
    return -1;
  } else {
    pci_read_config_dword(dev,data,&data2);
    return data2;
  }
}

//*************************************************************************
void bestpci_write_config_dw(struct pci_dev *dev,b_configdrivertype *data) {
  bestpci_device_info *devinfo = pci_get_drvdata(dev);

  if (devinfo->f_using_direct_io) {
    return;
  } else {
    pci_write_config_dword(dev,data->Offset,data->Data.Data);
  }
}


/////////////////////////////////////////////////////////////////////////
// linux kernel version dependent calls
/////////////////////////////////////////////////////////////////////////
static void bestpci_create_class(void) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,13) // kernel 2.6.21 uses this
  bestpci_device_class = class_create(THIS_MODULE,"bestpci");
#else // kernel 2.6.9 use this
  bestpci_device_class = class_simple_create(THIS_MODULE,"bestpci");  
#endif
}

static void bestpci_destroy_class(void) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,13) // use this for 2.6.21
  class_destroy(bestpci_device_class);
#else // use this for 2.6.9
  class_simple_destroy(bestpci_device_class);
#endif
}

static struct class_device* bestpci_create_class_device(int minor) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,13) // kernel 2.6.21 uses this
  return class_device_create(bestpci_device_class, NULL, MKDEV(BESTPCI_MAJOR,minor), NULL, "bestpci%d",minor);
#else // use this for 2.6.9
  return class_simple_device_add(bestpci_device_class, MKDEV(BESTPCI_MAJOR,minor), NULL, "bestpci%d",minor);
#endif
}

static void bestpci_destroy_class_device(int minor) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,13) // kernel 2.6.21 uses this
  class_device_destroy(bestpci_device_class,MKDEV(BESTPCI_MAJOR,minor));
#else // use this for 2.6.9
  class_simple_device_remove(MKDEV(BESTPCI_MAJOR,minor));
#endif
}


module_init(bestpci_init_module);
module_exit(bestpci_exit_module);
